﻿Imports System.Drawing

Public Class MapForm
    Private treasures As New List(Of Point)           ' active treasures (not yet found) - in Form1 coordinates
    Private foundTreasurePoints As New List(Of Point) ' found treasures - in Form1 coordinates
    Private playerClicks As New List(Of Point)        ' player click points - in Form1 coordinates

    Private uiFont As New Font("Segoe UI", 12, FontStyle.Regular)
    Private treasureImages As New List(Of Image)
    Private rand As New Random()

    ' The source rectangle on Form1 where the map area lives (in Form1 client coordinates).
    Private sourceMapRect As Rectangle = Rectangle.Empty

    Public Sub New()
        InitializeComponent()
        Me.Text = "Treasure Map"
        Me.Size = New Size(320, 260)
        Me.BackColor = Color.Beige
        Me.DoubleBuffered = True
        Me.Font = uiFont

        ' --- Load small treasure icons if available ---
        Dim imageFiles As String() = {"coin.png", "gem.png", "chest.png"}
        For Each imgPath In imageFiles
            If IO.File.Exists(imgPath) Then
                treasureImages.Add(Image.FromFile(imgPath))
            End If
        Next
    End Sub

    ' NOTE: new signature includes sourceRect to allow correct scaling
    Public Sub UpdateMap(activeTreasures As List(Of Point),
                         foundTreasures As List(Of Point),
                         clickPoints As List(Of Point),
                         sourceRect As Rectangle)

        ' Store copies in original coordinate space (Form1's coordinates)
        treasures = If(activeTreasures, New List(Of Point)())
        foundTreasurePoints = If(foundTreasures, New List(Of Point)())
        playerClicks = If(clickPoints, New List(Of Point)())

        sourceMapRect = sourceRect
        Me.Invalidate()
    End Sub

    Protected Overrides Sub OnPaint(e As PaintEventArgs)
        MyBase.OnPaint(e)
        Dim g As Graphics = e.Graphics
        g.Clear(Me.BackColor)

        ' If the source map rect is empty, we can't scale properly — show a friendly message
        If sourceMapRect = Rectangle.Empty OrElse sourceMapRect.Width <= 0 OrElse sourceMapRect.Height <= 0 Then
            Using drawFont As New Font("Segoe UI", 10, FontStyle.Italic)
                g.DrawString("Map unavailable (invalid map bounds).", drawFont, Brushes.Black, 10, 10)
            End Using
            Return
        End If

        ' Compute scaling from Form1 map area -> this MapForm client area.
        Dim destRect As New Rectangle(10, 10, Me.ClientSize.Width - 20, Me.ClientSize.Height - 20)
        If destRect.Width <= 0 OrElse destRect.Height <= 0 Then Return

        ' Keep aspect ratio by fitting source into dest
        Dim scaleX As Double = destRect.Width / sourceMapRect.Width
        Dim scaleY As Double = destRect.Height / sourceMapRect.Height

        ' Optional: choose uniform scale to avoid distortion; uncomment if desired:
        ' Dim scale As Double = Math.Min(scaleX, scaleY)
        ' scaleX = scale
        ' scaleY = scale

        ' Origin offsets for the destination drawing (we inset by 10 px for padding)
        Dim offsetX As Integer = destRect.X
        Dim offsetY As Integer = destRect.Y

        ' Helper to convert a point from source (Form1) to map client coords
        Dim toMapPoint = Function(p As Point) As Point
                             Dim sx As Integer = CInt((p.X - sourceMapRect.X) * scaleX) + offsetX
                             Dim sy As Integer = CInt((p.Y - sourceMapRect.Y) * scaleY) + offsetY
                             Return New Point(sx, sy)
                         End Function

        ' Draw map background box
        g.DrawRectangle(Pens.Brown, destRect)

        ' Draw unfound treasures (use small icons or gold dots)
        For Each t In treasures
            Dim mp As Point = toMapPoint(t)
            If treasureImages.Count > 0 Then
                Dim imgSize As Integer = 12
                Dim img As Image = treasureImages(rand.Next(treasureImages.Count))
                g.DrawImage(img, mp.X - imgSize \ 2, mp.Y - imgSize \ 2, imgSize, imgSize)
            Else
                g.FillEllipse(Brushes.Gold, mp.X - 4, mp.Y - 4, 8, 8)
                g.DrawEllipse(Pens.Brown, mp.X - 4, mp.Y - 4, 8, 8)
            End If
        Next

        ' Draw found treasures (green markers)
        For Each p In foundTreasurePoints
            Dim mp As Point = toMapPoint(p)
            g.FillEllipse(Brushes.LightGreen, mp.X - 4, mp.Y - 4, 8, 8)
            g.DrawEllipse(Pens.DarkGreen, mp.X - 4, mp.Y - 4, 8, 8)
        Next

        ' Draw player click markers ("X") — red X slightly offset for clarity
        Using redPen As New Pen(Color.Red, 2)
            For Each p In playerClicks
                Dim mp As Point = toMapPoint(p)
                g.DrawLine(redPen, mp.X - 6, mp.Y - 6, mp.X + 6, mp.Y + 6)
                g.DrawLine(redPen, mp.X + 6, mp.Y - 6, mp.X - 6, mp.Y + 6)
            Next
        End Using

        ' Optional: draw a small legend
        Using smallFont As New Font("Segoe UI", 8, FontStyle.Regular)
            Dim lx As Integer = destRect.Right - 100
            Dim ly As Integer = destRect.Bottom - 40
            g.DrawString("Legend:", smallFont, Brushes.Black, lx, ly)
            g.FillEllipse(Brushes.Gold, lx + 45, ly, 8, 8) : g.DrawString("Unfound", smallFont, Brushes.Black, lx + 60, ly - 2)
            g.FillEllipse(Brushes.LightGreen, lx + 45, ly + 12, 8, 8) : g.DrawString("Found", smallFont, Brushes.Black, lx + 60, ly + 10)
            g.DrawLine(Pens.Red, lx + 47, ly + 26, lx + 55, ly + 34) : g.DrawString("Clicks", smallFont, Brushes.Black, lx + 60, ly + 22)
        End Using
    End Sub
End Class
